using System.Collections;
using UnityEngine;
using MathNet.Numerics.Distributions;
using UnityEngine.SceneManagement;
using System.IO;

/*
 * Manager Script for the Non-Visualised Liquid Democracy Simulation Environment (Non-visualised version of LiquidDemocracy.cs)
 */
public class NonSimulationScript : MonoBehaviour
{
    int numRuns;
    int run = 0;
    bool oneShot;

    int numVoters;
    public VoterScript voterPrefab;
    VoterScript[] voters;

    StreamWriter csvWriter;

    // Start is called before the first frame update
    void Start()
    {
        // .csv output folder and file set up, simulation put in start state
        string folder = "TestData";
        if (!Directory.Exists(folder)) { Directory.CreateDirectory(folder); }
        csvWriter = new StreamWriter("TestData/runData.csv");

        csvWriter.WriteLine("Run,Correct Votes,Incorrect Votes,#Gurus,Highest Power,Avg Vote Comp");

        numVoters = StatsManager.instance.numVoters;
        numRuns = StatsManager.instance.numRounds;

        if (StatsManager.instance.roundType == 0) { oneShot = true; }
        else { oneShot = false; }

        voters = new VoterScript[numVoters];
        StartCoroutine(Spawn());
    }


    // One Shot: Simulation moved on to next round or ends
    void NextRun()
    {
        run++;
        StatsManager.instance.round = run;

        if (run < numRuns)
        {
            foreach (VoterScript voter in voters)
            {
                voter.Remove();
            }

            voters = new VoterScript[numVoters];
            StartCoroutine(Spawn());
        }
        else
        {
            EndSimulation();
        }
    }

    // Learning Agents: Simulation moves on to next vote or ends
    void NextVote()
    {
        run++;
        StatsManager.instance.round = run;

        if (run < numRuns)
        {
            VoterScript[] votersByComp = new VoterScript[numVoters];
            System.Array.Copy(voters, votersByComp, numVoters);
            System.Array.Sort(votersByComp, delegate (VoterScript x, VoterScript y) { return x.percievedCompetence.CompareTo(y.percievedCompetence); });
            System.Array.Reverse(votersByComp);

            foreach (VoterScript voter in voters)
            {
                if (voter.IsGuru())
                {
                    voter.ReviewVote(votersByComp);
                }
            }

            StartCoroutine(Votes());
        }
        else
        {
            EndSimulation();
        }
    }

    // Spawn the desired number of voters 
    IEnumerator Spawn()
    {

        int voterID = 0;
        var gammaDistribution = new Gamma(1.0, 1.0 / StatsManager.instance.compGain);

        int rowSize;
        if (numVoters < 30) { rowSize = 5; }
        else if (numVoters < 60) { rowSize = 10; }
        else { rowSize = 20; }

        int numRows = (int)System.Math.Ceiling((float)numVoters / rowSize);
        float xStartPos = 3 * -((rowSize / 2.0f) - 0.5f);
        float zStartPos = 3 * ((numRows / 2.0f) - 0.5f);

        for (int i = 0; i < numRows; i++)
        {
            int rowVoters = rowSize;
            if (i == numRows - 1 && numVoters % rowSize != 0)
            {
                rowVoters = numVoters % rowSize;
            }
            for (int j = 0; j < rowVoters; j++)
            {
                voters[voterID] = Instantiate(voterPrefab, new Vector3(xStartPos + (j * 3.0f), 2.9f, zStartPos - (i * 3.0f)), Quaternion.identity);

                double gammaComp = gammaDistribution.Sample();

                voters[voterID].SetupVoter(voterID, gammaComp, StatsManager.instance.speed, StatsManager.instance.prefAttach, StatsManager.instance.compImp, StatsManager.instance.alphaValue);
                voterID++;
            }
            yield return null;
        }

        StartCoroutine(Delegations());
    }

    // Voters make their first guru and delegator decision, and make delegations if they choose to
    IEnumerator Delegations()
    {
        foreach (VoterScript voter in voters)
        {
            voter.SetupNetwork(StatsManager.instance.networkSize, voters);
        }

        VoterScript[] votersByComp = new VoterScript[numVoters];
        System.Array.Copy(voters, votersByComp, numVoters);

        System.Array.Sort(votersByComp, delegate (VoterScript x, VoterScript y) { return x.percievedCompetence.CompareTo(y.percievedCompetence); });
        System.Array.Reverse(votersByComp);

        for (int i = 0; i < numVoters; i++)
        {
            voters[i].GuruOrDelegator(votersByComp);
            yield return null;
        }

        StartCoroutine(Votes());
    }

    // Gurus make their votes
    IEnumerator Votes()
    {
        int right = 0;
        int wrong = 0;

        int highestPower = 0;
        int numGurus = 0;
        double avgCompetence = 0;

        for (int i = 0; i < numVoters; i++)
        {
            if (voters[i].IsGuru())
            {
                numGurus++;

                int vote;
                int power;

                voters[i].CastVote(out vote, out power);
                if (vote == 0) { wrong += power; }
                else { right += power; }

                if(power > highestPower) { highestPower = power; }
                avgCompetence += voters[i].GetTrueCompetence() * power;
            }
            yield return null;
        }

        avgCompetence = avgCompetence / numVoters;

        int runNum = run + 1;
        csvWriter.WriteLine(runNum + "," + right + "," + wrong + "," + numGurus + "," + highestPower + "," + avgCompetence);


        //In case of draw, coin flip to decide winning decision
        if (right == wrong) { 
            int coinFlip = Random.Range(0, 2);
            if (coinFlip == 0)
            {
                wrong++;
            }
            else
            {
                right++;
            }

        }

        if (StatsManager.instance.roundType == 1)
        {
            foreach (VoterScript voter in voters)
            {
                voter.UpdatePercievedComp();
            }
        }

        if (right > wrong)
        {
            StatsManager.instance.correct++;
        }
        else
        {
            StatsManager.instance.incorrect++;
        }
          
        StatsManager.instance.numGurusTotal += numGurus;
        StatsManager.instance.avgCompetenceTotal += avgCompetence;
        StatsManager.instance.highestPowerTotal += highestPower;


        if (oneShot)
        {
            NextRun();
        }
        else
        {
            NextVote();
        }
    }

    // Simulation reaches the end, summary of results added to CSV file, statistics scene loaded
    void EndSimulation()
    {
        // Write to CSV
        csvWriter.WriteLine();
        csvWriter.WriteLine("Summary");
        csvWriter.WriteLine("Correct Decisions,Incorrect Decisions,Avg #Gurus,Avg Highest Power,Avg Vote Comp");
        csvWriter.WriteLine(StatsManager.instance.correct + "," + StatsManager.instance.incorrect + "," + StatsManager.instance.numGurusTotal/ StatsManager.instance.numRounds + "," +
            StatsManager.instance.highestPowerTotal / StatsManager.instance.numRounds + "," + StatsManager.instance.avgCompetenceTotal / StatsManager.instance.numRounds);
        csvWriter.Close();

        // Load stats scene
        StatsManager.instance.running = false;
        SceneManager.LoadScene("StatsScene");
    }
}